const GrantApplication = require('../models/GrantApplication');

// @desc    Submit a new grant application
// @route   POST /api/grants/apply
// @access  Public (or Private if using auth)
const submitApplication = async (req, res) => {
  try {
    const { 
      applicationType, 
      fullName, 
      idNumber, 
      companyName, 
      ein, 
      projectDescription 
    } = req.body;

    // 1. Basic Validation
    if (!applicationType || !projectDescription) {
      return res.status(400).json({ message: 'Missing required fields' });
    }

    // 2. Construct the data object
    const grantData = {
      applicationType,
      projectDescription,
      status: 'pending' // Default status
    };

    // 3. Add type-specific data
    if (applicationType === 'individual') {
        if (!fullName || !idNumber) {
            return res.status(400).json({ message: 'Individual applications require Name and ID Number' });
        }
        grantData.fullName = fullName;
        grantData.idNumber = idNumber;
    } 
    else if (applicationType === 'company') {
        if (!companyName || !ein) {
            return res.status(400).json({ message: 'Company applications require Company Name and EIN' });
        }
        grantData.companyName = companyName;
        grantData.ein = ein;
    } else {
        return res.status(400).json({ message: 'Invalid application type' });
    }

    // 4. Create and Save
    const newGrant = await GrantApplication.create(grantData);

    res.status(201).json({
      success: true,
      message: 'Application submitted successfully',
      data: newGrant
    });

  } catch (error) {
    console.error('Error submitting grant:', error);
    res.status(500).json({ 
      success: false, 
      message: 'Server Error: Could not submit application',
      error: error.message 
    });
  }
};

// @desc    Get all applications (For Admin Dashboard)
// @route   GET /api/grants
const getApplications = async (req, res) => {
    try {
        const grants = await GrantApplication.find().sort({ createdAt: -1 });
        res.status(200).json(grants);
    } catch (error) {
        res.status(500).json({ message: error.message });
    }
};

module.exports = {
  submitApplication,
  getApplications
};